<?php
/**
 * Инициализация шаблона
 *
 * 
 * 1. Подключение перевода
 * 2. Добавление .html на статических страницах и товарах
 * 3. Отключение перенаправления канонических ссылок
 * 4. Включение поддержки HTML5
 * 5. 
 *
 * @param string $set_theme_name
 *  
 * @author veselka.ua
 * @version 0.4
 *
 * @package veselka_ua/themes
 */

class WpThemeInit {

	/**
	 * Имя шаблона.
	 * @type array
	 */
	private $theme_name = false;

	public function __construct( $set_theme_name = false ) {

		// Временная зона
		$timezone = 'Europe/Kiev';
		if( (bool)get_option('public_timezome') ) {
			$timezone = get_option('public_timezome');
		}
		date_default_timezone_set($timezone);

		// Имя шаблона
		if( $set_theme_name ) {
			$this->theme_name = $set_theme_name;
		}
		if( !$this->theme_name ) {
			$theme_data = wp_get_theme();
			$this->theme_name = $theme_data['Name'];
		}
		/**
		 * Разрешить html в цитатах.
		 */
		remove_filter('get_the_excerpt', 'wp_trim_excerpt');
		/**
		 * Подключение переводов.
		 */
		add_action( 'init', [$this, 'load_theme_textdomain'], 10, 0 );
		/**
		 * .html на страницах.
		 */
		add_action( 'init', [$this, 'change_page_permalink'], 20, 0);
		/**
		 * Настройки шаблона каждый раз при включении.
		 */
		add_action( 'after_setup_theme', [ $this, 'get_veselka_setup' ], 10 );
		/**
		 * .html на страницах произвольних типов записей.
		 */
		add_action( 'rewrite_rules_array', [$this, 'rewrite_rules'] );
		add_filter( 'post_type_link', [$this, 'custom_post_permalink'], 1, 2 );
		/**
		 * Отключаем автосохранение.
		 */
		add_action( 'wp_print_scripts', [$this, 'disable_autosave'] );
		/**
		 * Остановить перенаправление канонические URL.
		 */
		add_filter( 'redirect_canonical', '__return_false' );
		/**
		 * Цитата в страницах.
		 */
		add_post_type_support( 'page', 'excerpt' );
		/**
		 * Отключение автоматических абзацев где это требуется.
		 */

		remove_filter( 'the_content', 'wpautop' );
		add_filter('the_content', [$this, 'custom_wpautop'], 20);
	}



	/**
	 * Подключение перевода.
	 *
	 * @param void
	 * @return void
	 */
	final public function load_theme_textdomain() {
		$path = '/lang';
		if( is_admin() ) {
			$path .= '/admin';
		}
		load_theme_textdomain( 'spiral', TEMPLATEPATH . $path );

	}


	/**
	 * Параметры шаблона при загрузке.
	 *
	 * @param void
	 * @return void
	 */
	final public function get_veselka_setup() {
		// Переключение стандартной верстки на поддерживающую HTML5
		add_theme_support( 'html5', [ 'search-form', 'comment-form', 'comment-list' ] );
	}


	/**
	 * .html на страницах.
	 *
	 * @param void
	 * @return void
	 */
	final public function change_page_permalink() {

		global $wp_rewrite;
		if( strstr($wp_rewrite->get_page_permastruct(), '.html') != '.html' ) {
			$wp_rewrite->page_structure = $wp_rewrite->page_structure . '.html';
		}
	}


	/**
	 * .html на страницах товаров.
	 *
	 * @param string $rules
	 * @return string
	 */
	final public function rewrite_rules( $rules ) {

		$new_rules = [];
		foreach ( get_post_types() as $type ) {
			$new_rules[ $type . '/([^/]+)\.html$' ] = 'index.php?post_type=' . $type . '&name=$matches[1]';
		}
		return $new_rules + $rules;

	}

	final public function custom_post_permalink( $post_link, $post ) {

		//global $post;
		$type = get_post_type( $post->ID );
		return home_url( $type . '/' . $post->post_name . '.html' );

	}


	/**
	 * Отключаем автосохранение страниц.
	 *
	 * @param void
	 * @return void
	 */
	final public function disable_autosave(){
		wp_deregister_script('autosave');
	}


	/**
	 * Отключение автоматических абзацев в страницах где указано доп. поле wpautop.
	 *
	 * @param string $content
	 * @return string
	 */
	final public function custom_wpautop($content) {
		// Указано доп поле wpautop
		if( get_post_meta(get_the_ID(), 'wpautop', true) == 'false' ) {
			// Удаление пустых абзацев, кроме всего прочего
			$content = str_replace( array( '<p></p>' ), '', $content );
			$content = str_replace( array( '<p>  </p>' ), '', $content );
			// Возвращаем контент как есть
			return $content;
		}
		// Фильтр
		return wpautop($content);
	}


} // end WpThemeInit
?>