<?php
/**
 * Формирование виджетов.
 *
 * Вспомогательный родительский класс содержащий общие элементы для всех виджетов.
 *
 * @author veselka.ua
 * @version 0.1
 *
 * @package veselka_ua/themes
 */

abstract class WidgetsKernel extends WP_Widget {

	use FileHandling;

	/**
	 * Регистр.
	 * @type array
	 */
	protected $registry = array();


//////////////////// Публичные методы ////////////////////

	/**
	 * Построение виджета.
	 *
	 * @param void
	 * @return void
	 */
	final public function widget( $args, $instance ) {
        extract( $args );
		if( empty($instance) ) {
			$instance = $this->registry['defaults'];
		}
        
        $title = apply_filters('widget_title', $instance['title']);

		// Подключение файла с отображением виджета
		require_once $this->registry['files']['view'];
	}


	/**
	 * Форма виджета в админпанели.
	 *
	 * @param void
	 * @return void
	 */
	final public function form( $instance ) {
		// Склейка параметров виджета
		$instance = wp_parse_args( (array)$instance, $this->registry['defaults'] );

		// Подключение файла с параметрами
		require_once $this->registry['files']['admin'];
	}


//////////////////// Защищенные методы ////////////////////

	/**
	 * Обработка входной информации.
	 *
	 * @param void
	 * @return void
	 */
	protected function load() {

		// Подготовка к работе, установка флагов в обычный режим работы
		$this->set_variables();

		// Проверка наличия требуемых файлов
		$this->check_required_files();

		// Получение параметров видежета
		$this->get_widget_parameters();
	}


	/**
	 * Регистрация виджета.
	 *
	 * @param void
	 * @return void
	 */
    protected function widget_register() {
		$widget_options = array('description' => $this->registry['meta']['description'] );
		$control_options = array( 'width' => $this->registry['meta']['width'] );
		$title = '&Xi; ' . $this->registry['meta']['title'];
		// Созданеие виджета
		parent::__construct( $this->registry['widget_slug'], $title, $widget_options, $control_options);
    }



//////////////////// Приватные методы ////////////////////

	/**
	 * Определение переменных.
	 *
	 * @param void
	 * @return void
	 */
	private function set_variables() {

		// Обязательные элементы виджета
		$this->registry['view'][] = 'admin';
		$this->registry['view'][] = 'view';
		$this->registry['view'][] = 'defaults';

		// Метаданные виджета
		$this->registry['widget_name'] = 'Veselka' . $this->registry['name'];
		$this->registry['widget_title'] = 'Veselka ' . $this->registry['name'];
		$this->registry['name'] = strtolower($this->registry['name']);
		$this->registry['widget_slug'] = 'veselka_' . $this->registry['name'];

		// Переключить флаги - указать что режим работы нормальный по умолчанию
		$this->registry['show'] = true; // Показывать виджет
	}


	/**
	 * Проверка наличия требуемых для работы файлов.
	 *
	 * Если файл существует путь записывается в массив.
	 * Дальнейшее обращение к файлу через элемент массива с индексом равным имени фала.
	 *
	 * @param void
	 * @return void
	 */
	private function check_required_files() {

		// Путь к файлу
		$file_name = $this->registry['name'] . '.php';

		foreach( $this->registry['view'] as $part_name ) {
			// Проверка наличия файла
			$this->registry['files'][$part_name] = $this->check_file_path( $file_name, 'widget_' . $part_name, true );
			if( !$this->registry['files'][$part_name] ) {
				$this->registry['show'] = false;
				return true;
			}
		}
		unset($this->registry['view']);
	}


	/**
	 * Получение параметров видежета.
	 *
	 * @param void
	 * @return void
	 */
	private function get_widget_parameters() {

		// Подключение файла с параметрами
		require_once $this->registry['files']['defaults'];

		// Исключения
		if( !isset($defaults) && !empty($defaults) ) {
			$this->registry['show'] = false;
			return true;
		}

		// Умолчания
		$this->registry['defaults'] = $defaults;
		// Метаданные виджета
		if( isset($meta) ) {
			$this->registry['meta'] = $meta;
		}
	}


	/**
	 * Обрезает текст по заданному размеру.
	 *
	 * @param string $string
	 * @param int $wordsreturned
	 * @return string
	 */
	protected function shorten( $string, $wordsreturned ) {
		$retval = $string;
		$array = explode( " ", $string );
		if( count($array) <= $wordsreturned ){
			$retval = $string;
		} else {
			array_splice($array, $wordsreturned);
			$retval = implode( " ", $array );
		}
		return $retval;
	}


	/**
	 * Удаляет шорткоды галереи.
	 *
	 * @param string $content
	 * @return string
	 */
	protected function strip_shortcode_gallery( $content ) {
		preg_match_all( '/'. get_shortcode_regex() .'/s', $content, $matches, PREG_SET_ORDER );
		if ( ! empty( $matches ) ) {
			foreach ( $matches as $shortcode ) {
				if ( 'gallery' === $shortcode[2] ) {
					$pos = strpos( $content, $shortcode[0] );
					if ($pos !== false)
						return substr_replace( $content, '', $pos, strlen($shortcode[0]) );
				}
			}
		}
		return $content;
	}

} // end WidgetsKernel