<?php
/**
 * Импорт параметров Таксономии.
 *
 * Импортирует параметры из Google Spreadsheets в базу данных WordPress.
 *
 * @param obj $gdrive
 *
 * @author veselka.ua
 * @version 0.1
 *
 * @package veselka_ua/themes
 */

class GdriveImportTax extends GdriveImportCommon {


	public function __construct( &$gdrive ) {

		// Объект инструментов Google Spreadsheets
		if( !$gdrive && !is_object($gdrive) ) {
			throw new \Exception('No Google tools obj');
		}
 
		// Запись объекта в свойство
		$this->gdrive = $gdrive;
		// Флаг
		$this->registry['single'] = false;
	}


/////////////////// Публичные методы ////////////////////


	/**
	 * Обертка для получения ключа.
	 *
	 * @param string $destination
	 * @return void
	 */
	final public function import_data( $destination='' ) {
		// Имя сущности
		$this->registry['destination'] = $destination;
		// Развертывание (Метод в родительском объекте)
		$this->load();
		// Перебор строк
		$this->parse_rows();
		// Запись обновленной таблицы в файл
		$this->gdrive->write_file_data();
	}


//////////////////// Приватные методы ////////////////////


	/**
	 * Перебор строк файла.
	 *
	 * @param void
	 * @return void
	 */
	private function parse_rows() {
		// Перебор значений
		while( list($row, $data) = $this->gdrive->get_file_data() ) {

			// Обнуление переменных
			$this->registry['up_to_date'] = false;
			$this->registry['tax'] = true;
			$this->registry['data'] = [];
			// Текущая строка
			foreach($this->index as $associative_key => $numeric_key ) {
				$this->registry['data'][$associative_key] = $data[$numeric_key];
			}
			$this->registry['data']['import_status'] = '';
			// Запуск методов в зависимости от состава файла загрузки
			foreach( $this->registry['methods'] as $name => $arg ) {
				// Формирование имени метода
				$method_name = $name . '_field_construct';
				// Вызов соответствующего метода для сбора данных
				if( method_exists( $this, $method_name ) ) {
					if( is_array($arg) && !empty($arg) ) {
						$this->{$method_name}($arg);
					} else {
						$this->{$method_name}();
					}
				}
			}

			// Запись результата
			if( $this->registry['tax'] ) {
				$this->create_record();
			}

			// Запись статуса
			$this->write_status();
			// Запись обновленных значений 
			foreach($this->index as $associative_key => $numeric_key ) {
				// Перевод в целочисленный формат
				if( in_array($associative_key, ['term_id','wp_id','tax_id','wp_order']) ) {
					$data[$numeric_key] = (int)$this->registry['data'][$associative_key];
				} else {
					$data[$numeric_key] = $this->registry['data'][$associative_key];
				}
			}
			// Запись в файл обновленной строки
			$this->gdrive->set_file_data($row, $data);
		}
	}


	/**
	 * Получение данных по ярлыку.
	 *
	 * @param array $data
	 * @return void
	 */
	private function slug_field_construct() {

		// Проверка ярлыка
		if( empty($this->registry['data']['term_id']) ) {
			if( empty($this->registry['data']['slug']) ) {
				// Параметры не переданы, новая запись
				$this->load_staus('status_new_slug');
				// Если существует метод описывающий генерацию ярлыка
				$method_name = $this->registry['destination'] . '_slug_construct';
				// Вызов соответствующего метода для сбора данных
				if( method_exists( $this, $method_name ) ) {
					$this->{$method_name}();
				} else {
					// Генератор ярлыка по умолчанию (транслитерация заголовка или h1)
					if( !empty($this->registry['data']['h1']) ) {
						$this->registry['data']['slug'] = apply_filters( 'url_transliteration', $this->registry['data']['h1'] );
					} elseif( !empty($this->registry['data']['title']) ) {
						$this->registry['data']['slug'] = apply_filters( 'url_transliteration', $this->registry['data']['title'] );
					}
				}
				if( empty($this->registry['data']['slug']) ) {
					// Slug не сгенерирован
					$this->load_staus('error');
					$this->load_staus('warn_no_slug_gen');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['tax'] = false;
				}
			} else {
				// Восстановление id из slug
				$tax = get_term_by('slug', strtolower((string)$this->registry['data']['slug']), $this->registry['destination']);

				if( is_object($tax) ) {
					$this->registry['tax'] = $tax;
					$this->registry['data']['term_id'] = $this->registry['tax']->term_id;
					// Запись id в массив добавления записи
					$this->registry['wp_tax']['term_id'] = $this->registry['tax']->term_id;
					// Запись загловка в массив добавления записи
					//$this->registry['wp_tax']['args']['name'] = $this->registry['tax']->name;
				} else {
					// Нет такого ярлыка, новая запись
					$this->load_staus('status_new');
					// Ярлык
					$this->registry['wp_tax']['args']['slug'] = $this->registry['data']['slug'];
					//$this->registry['wp_tax']['term'] = $this->registry['data']['name'];
				}
			}
		} else {
			$this->registry['tax'] = get_term($this->registry['data']['term_id'], $this->registry['destination']);
			if( empty($this->registry['data']['slug']) ) {
				// Восстановление slug из id
				$this->registry['data']['slug'] = $this->registry['tax']->slug;
			} else {
				// Оба параметра переданы, сравнение
				// Восстановление id из slug для проверки
				$tax = get_term_by('slug', strtolower((string)$this->registry['data']['slug']), $this->registry['destination']);
				if( $tax->term_id != $this->registry['data']['term_id'] ) {
					// Slug ротиворечит ID
					$this->load_staus('error');
					$this->load_staus('warn_bad_slug');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['tax'] = false;
				}
			}
			// Запись id в массив добавления записи
			$this->registry['wp_tax']['term_id'] = $this->registry['tax']->term_id;
			// Запись ярлыка в массив добавления записи
			$this->registry['wp_tax']['args']['slug'] = $this->registry['tax']->slug;
			// Запись загловка в массив добавления записи
			//$this->registry['wp_tax']['args']['name'] = $this->registry['tax']->name;
		}

	}


	/**
	 * Проверка актуальности записи.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function date_field_construct() {
		// Нет даты модификации
		if( $this->registry['data']['modification_time'] == '' ) {
			return true;
		}

		// Объекта нет
		if( !is_object($this->registry['tax']) ) {
			return true;
		}
		// Дата из файла
		$file_timestamp = strtotime($this->registry['data']['modification_time']);
		$file_date = date('Y-m-j H:i:s',$file_timestamp);
		// Сравнение даты из базы
		$t_id = $this->registry['tax']->term_id;
		$tax_data = get_option('taxonomy_'.$t_id);
		$db_timestamp = $tax_data['cat_modify'];
		$db_timestamp = strtotime($db_timestamp);
		// Сравнение дат
		if( $file_timestamp > $db_timestamp ) {
			// Данные будут обновлены
			$this->registry['data']['cat_modify'] = ($this->registry['data']['modification_time']);
		} else if( $file_timestamp == $db_timestamp ) {
			// Пост изменен позднее, не пишем в базу
			$this->load_staus('status_actual');
			$this->registry['up_to_date'] = true;
		} else {
			// Пост изменен позднее, не пишем в базу
			$this->load_staus('warning');
			$this->load_staus('warn_old_date');
			// Уничтожение объекта - пеерход к следующей строке
			$this->registry['tax'] = false;
		}
	}


	/**
	 * Создание таксономии.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function create_record() {
		// Нужно ли обновление
		if( $this->registry['up_to_date'] ) {
			return NULL;
		}

		// Параметры для создания/обновления записи
		$this->registry['wp_tax']['taxonomy'] = $this->registry['destination'];
		$this->registry['wp_tax']['args']['parent'] = 0;
		//$this->registry['wp_tax']['args']['description'] = '';
		if( isset($this->registry['data']['description']) && !empty($this->registry['data']['description']) ) {
			$this->registry['wp_tax']['args']['description'] = $this->registry['data']['description'];
		}
		$this->registry['wp_tax']['args']['name'] = $this->registry['data']['name'];
		// Есть родительская запись
		if( $this->registry['data']['parent'] != '' ) {
			$parent = get_term_by('slug', strtolower((string)$this->registry['data']['parent']), $this->registry['destination']);
			if( !is_wp_error($parent) ) {
				$this->registry['wp_tax']['args']['parent'] = $parent->term_id;
			}
		}

		// Отмена фильтрации контента категорий
		remove_filter( 'pre_term_description', 'wp_filter_kses' );
		remove_filter( 'term_description', 'wp_kses_data' );
		// Создать или обновить запись
		if( (bool)$this->registry['wp_tax']['term_id'] ) {
			$new = wp_update_term($this->registry['wp_tax']['term_id'],$this->registry['wp_tax']['taxonomy'],$this->registry['wp_tax']['args']);
			$this->load_staus('status_update');
		} else {
			$new = wp_insert_term($this->registry['wp_tax']['term'],$this->registry['wp_tax']['taxonomy'],$this->registry['wp_tax']['args']);
			$this->load_staus('status_create');
		}
		// Проверка записи
		if( is_wp_error($new) ) {
			// Не удалось создать термин
			$this->load_staus('error');
			$this->load_staus('warn_write_err');
			$this->registry['tax'] = false;
			return NULL;
		} else {
			$this->registry['wp_tax']['term_id'] = $new['term_id'];
		}

		// Мета данные
		$term_id = $this->registry['wp_tax']['term_id'];
		//$tax_data = get_option("taxonomy_$term_id");5
		$tax_data = get_term_meta( $term_id );

		// Поиск значений для удаления
		foreach( $tax_data as $meta_key => $meta_value ) {
			if( $meta_value == '' ) {
				delete_term_meta( $term_id, $meta_key );
			}
		}

		// Обработка старых значений (Записанных в дугую таблицу)
		$old_cat_meta = get_option("taxonomy_$term_id");
		if( isset($old_cat_meta) && !empty($old_cat_meta) ) {
			foreach( $old_cat_meta as $meta_key => $meta_value ) {
				// Создать/Обновить/Удалить
				if( $meta_value != '' ) {
					update_term_meta( $term_id, $meta_key, $meta_value );
				}
			}
			// Удаление значений
			update_option("taxonomy_$term_id", []);
		}

		// Обновление метаданных
		foreach( $this->registry['data'] as $meta_key => $meta_value ) {
			// Мета данные?
			$cat = explode('_',$meta_key);
			if( $cat[0] != 'cat' ) {
				continue;
			}
			// Создать/Обновить/Удалить
			if( $meta_value == '' ) {
				delete_term_meta( $term_id, $meta_key );
			} else {
				update_term_meta( $term_id, $meta_key, $meta_value );
			}
		}

/*		if( empty($tax_data) ) {
			// Нет данных для обновления
			$this->load_staus('warning');
			$this->load_staus('warn_empty_meta');
		} else {

			// Не удалось обновить метаданные термина 
			$this->load_staus('warning');
			$this->load_staus('warn_db_equal');
		}*/
	}


} //end GdriveImportTax