<?php
/**
 * Импорт параметров в базу WP.
 *
 * Общие инструменты импорта для таксономий и единичных записей.
 *
 * @param void
 *
 * @author veselka.ua
 * @version 0.1
 *
 * @package veselka_ua/themes
 */

class GdriveImportCommon {


	use PluginsPath;


	/**
	 * Регистр.
	 *
	 * single		|	bool	- Импорт единичных записей
	 * destination	|	string	- Значение аргумента объекта.
	 * option_name	|	string	- Наименования параметра GDrive с реквизитами доступа.
	 * current_path	|	string	- Путь к файлу с параметрами.
	 * file_name	|	string	- Наименование файла параметров (эталонных значений мета данных).
	 * data			|	array	- Массив (ассоциативный) с текущими значениями строки.
	 * methods		|	array	- Массив префиксов для формирования методов обработки данных.
	 * wp_post		|	array	- Массив аргументов для создания/изменения записи в бд.
	 * status		|	string	- Статус загрузки текущей строки.
	 * post			|	obj		- Объект текущего поста, чтобы не вызывать много раз.
	 * tax			|	string	- Наименование таксономии.
	 * 
	 *
	 *
	 * @type array
	 */
	protected $registry = [];
	/**
	 * Инструменты работы с Google Spreadsheets.
	 * @type obj
	 */
	protected $gdrive;
	/**
	 * Массив указателей.
	 * Используется для поиска по массиву данных, чтобы не превращать масивы данных в ассоциативные.
	 * @type array
	 */
	protected $index;
	/**
	 * Эталонный массив значений таблицы.
	 * @type array
	 */
	protected $etalon;
	/**
	 * Массив сообщений.
	 * @type array
	 */
	protected $msg;

	/**
	 * Развертывание.
	 *
	 * @param void
	 * @return void
	 */
	protected function load() {

		// Получение массива сообщений
		$this->get_msgs();


		// Проверка наличия плагина
		// Плагины могут быть дочерними (частными случаями)
		// Имена констрант с путями могут совпадать для разных имен плагинов
		$this->check_plugins();

		// Получение наименований параметров
		$this->load_options();

		// Загрузка Google tools
		$this->gdrive->load_gdrive($this->registry['option_name']);
		// Получение идентификаторов полей
		$this->index = $this->gdrive->get_file_fields();

		// Запись эталонных значений
		$this->check_config_file();
		// Анализ массива идентификаторов
		$this->ids_analysis();

		// Массив идентификаторов больше не нужен, превращаем в массив указателей, элементы без указателей уничтожаются
		$this->index = array_flip($this->index);
	}


	/**
	 * Получение массива сообщений.
	 *
	 *
	 * @param string $add
	 * @return void
	 */
	protected function get_msgs() {
		/**
		 * Стандартный путь к параметрам.
		 */
		require TEMPLATEPATH . '/conf/gdrive.php';
		// Запись в свойство
		$this->msg = $msg;
	}


	/**
	 * Сохранение лога импорта строки.
	 *
	 *
	 * @param string $add
	 * @return void
	 */
	protected function load_staus($add) {
		if( !empty($this->registry['data']['import_status']) ) {
			if( strpos($add,'warn_') === 0 ) {
				$this->registry['data']['import_status'] .= ': ';
			} else {
				$this->registry['data']['import_status'] .= ' | ';
			}
		} else {
			$this->registry['data']['import_status'] = '';
		}
		$this->registry['data']['import_status'] .= $this->msg[$add];
	}


	/**
	 * Запись лога импорта строки.
	 *
	 *
	 * @param void
	 * @return void
	 */
	protected function write_status() {
		$status = $this->msg['ok'];
		if( !$this->registry['post'] && !$this->registry['tax'] ) {
			$status = $this->msg['fail'];
		}
		if( !empty($this->registry['data']['import_status']) ) {
			$status .= ' | ';
		}
		$this->registry['data']['import_status'] = $status . $this->registry['data']['import_status'];
	}


//////////////////// Приватные методы ////////////////////


	/**
	 * Получение наименований параметров.
	 *
	 * @param void
	 * @return void
	 */
	private function load_options() {

		// Исключение на случай отсутствия параметров
		$destination = $this->registry['destination'];
		if( !$destination || $destination == '' ) {
			$this->registry['destination'] = 'page';
		}

		// Конструктор имени параметра
		// Дополнение массивов, плагинами
		$this->plugins_tax = array_merge_recursive( ['wp'=>'page','category'=>'post'], $this->plugins_tax );
		$this->plugins_posts = array_merge_recursive( ['page'=>'page','post'=>'post'], $this->plugins_posts );
		$this->plugins_path = array_merge_recursive( ['page'=>TEMPLATEPATH,'post'=>TEMPLATEPATH], $this->plugins_path );

		// Полечения параметров доступа для единичных записей.
		if( isset($this->plugins_posts[$destination]) && !empty($this->plugins_posts[$destination]) ) {
			$key = $this->plugins_posts[$destination];
			$prefix = array_search( $key, $this->plugins_tax );
			$suffix = '_'.$destination;
			$file_prefix = 'single_meta_';
		// Полечения параметров доступа для таксономий.
		} else if( isset($this->plugins_tax[$destination]) && !empty($this->plugins_tax[$destination]) ) {
			$key = $this->plugins_tax[$destination];
			$prefix = $destination;
			$suffix = '';
			$file_prefix = 'tax_meta_';
		}
		$this->registry['tax'] = $prefix;
		// Наименования параметра GDrive
		$this->registry['option_name'] = $prefix . '_gdrive' . $suffix;
		// Наименование файла параметров
		$this->registry['file_name'] = $file_prefix . $destination . '.php';
		// Путь
		$this->registry['current_path'] = $this->plugins_path[$key] . '/conf/';
	}


	/**
	 * Проверка наличия конфигурационного файла.
	 * Запись эталонных значений
	 *
	 * @param void
	 * @return void
	 */
	private function check_config_file() {
		$filepath = $this->registry['current_path'] . $this->registry['file_name'];
		if( !file_exists($filepath) ) {
			throw new \Exception('No field options file' . PHP_EOL . $filepath . PHP_EOL);
		}
		// Запись эталонных значений
		require_once $filepath;
		if( isset($meta_fields) ) {
			$this->etalon = $meta_fields;
		}

		// Поля таксономии
		if( isset($cat_fields) ) {
			$this->etalon = $cat_fields;
			// Поля сортировки и SEO
			foreach( ['seo','sorting'] as $key ) {
				$filepath = TEMPLATEPATH . '/conf/tax_meta_' . $key . '.php';
				if( file_exists($filepath) ) {
					require_once $filepath;
					$this->etalon = array_merge_recursive( $this->etalon, $cat_fields );
				}
			}
		}
	}


	/**
	 * Анализ массива идентификаторов.
	 * Составление списка методов для обработки данного массива данных.
	 *
	 * @param void
	 * @return void
	 */
	private function ids_analysis() {
		// Нет ярлыка?
		if( in_array('wp_slug',$this->index) || in_array('slug',$this->index) ) {
			$this->registry['methods']['slug'] = true;
		} else {
			$this->registry['methods']['id'] = true;
		}
		// Обработка таксономий
		$this->registry['methods']['tax'] = true;
		// Дата модификации
		if( in_array('modification_time',$this->index) ) {
			$this->registry['methods']['date'] = true;
		}
		// Сортировка
		if( in_array('wp_order',$this->index) ) {
			$this->registry['methods']['sort'] = true;
		}
		// Сортировка
		if( in_array('title',$this->index) ) {
			$this->registry['methods']['title'] = true;
		}
		// Статус
		$this->registry['methods']['status'] = true;
		// Поиск параметров
		foreach( $this->index as $data ) {
			$prefix = explode('_',$data);
			if( in_array($prefix[0],['seo','meta']) ) {
				$sub = $prefix[0].'_';
				$sub = str_replace( $sub,'',$data);
				$this->registry['methods'][$prefix[0]][] = $sub;
			} else if( in_array($data,['h1','excerpt','content']) ) {
				$this->registry['methods'][$data] = true;
			}
		}

	}


} //end GdriveImportCommon