<?php
/**
 * Сбор данных форм.
 *
 * Общий объект.
 * 
 * Результаты работы данно класса:
 * 1. Структурированные POST данные.
 * 2. Имена полей из конфигурационных файлов.
 * 3. Метод сборки результата посредствам callback функции.
 *
 * TODO: Загружать формы из плагинов вместо custom_forms.
 *
 * @author veselka.ua
 * @version 0.1
 *
 * @package veselka_ua/themes
 */

class FormsData {


	use PluginsPath;

	/**
	 * Массив параметров.
	 * @type array
	 */
	private $options = [];

	/**
	 * Массив POST после обработки.
	 * @type array
	 */
	private $post_data = [];

	/**
	 * Список полей текущей контактной формы.
	 * @type array
	 */
	private $fields = [];

	/**
	 * Названия полей.
	 * @type array
	 */
	private $labels = [];

	/**
	 * Подсказки полей.
	 * @type array
	 */
	private $placeholders = [];

	/**
	 * Результат работы.
	 * @type mixed array|string
	 */
	private $result_set;


	public function __construct() {

		// Проверка наличия плагина
		// Плагины могут быть дочерними (частными случаями)
		// Имена констрант с путями могут совпадать для разных имен плагинов
		$this->check_plugins();

		// Инициализация
		$this->options['init'] = true;
	}


//////////////////// Публичные методы ////////////////////


	/**
	 * Геттер.
	 *
	 * @param string $name
	 * @return mixed
	 */
    final public function __get($name) {
		if( isset($this->$name) ) {
			return $this->$name;
		} else {
			return false;
		}
	}


	/**
	 * Инициализация.
	 *
	 * @param string $form_name
	 * @return void
	 */
	final public function get_load($form_name=false) {
		$this->load($form_name);
	}


	/**
	 * Инъекция POST переменной.
	 *
	 * @param string $name
	 * @param string $value
	 * @return void
	 */
	final public function inject_post_var($name,$value) {
		if( !isset($this->post_data[$name]) ) {
			$this->post_data[$name] = $value;
		}
	}


	/**
	 * Инъекция POST переменной.
	 *
	 * @param string $name
	 * @param string $value
	 * @return void
	 */
	final public function change_post_var($name,$value) {
		if( isset($this->post_data[$name]) ) {
			$this->post_data[$name] = $value;
		}
	}

	/**
	 * Обертка для получения обработанных данных.
	 *
	 *
	 * @param array $data
	 * @param calback $header_callback
	 * @param calback $line_callback
	 * @param calback $wrapper_callback
	 * @return mixed array|string $this->result_set
	 */
	final public function get_result( $data, $header_callback, $line_callback, $wrapper_callback ) {
		$this->load();
		$this->get_result_set( $data, $header_callback, $line_callback, $wrapper_callback );
		return $this->$result_set;
	}


//////////////////// Приватные и защищенные методы ////////////////////


	/**
	 * Развертывание.
	 *
	 * @param string $form_name
	 * @return void
	 */
	private function load($form_name) {

		// Уже загружалось
		if( !$this->options['init'] ) {
			return true;
		}

		// Параметры
		$this->get_options($form_name);

		// Получение и обработка входных данных
		$this->input_data_processing();

		// Получение текстовой информации (названия полей, подсказки)
		$this->get_txt_data();

		// Плучение параметров текущей контктной формы
		$this->get_form_data();

		// Инициализация завершена
		$this->options['init'] = false;

	}


	/**
	 * Параметры.
	 *
	 * @param string $form_name
	 * @return void
	 */
	private function get_options($form_name) {

		// Проверка наличия имени формы в массиве POST
		if( !$form_name && !isset($_POST['form_name']) ) {
			throw new Exception('No form specification!' );
		}

		$this->options['form_name'] = $form_name;
		if( isset($_POST['form_name']) ) {
			// Имя формы в формате css
			$this->options['form_name'] = htmlspecialchars($_POST['form_name']);
			unset($_POST['form_name']);
		}
		// Имя формы в формате php
		$this->options['array_name'] = str_replace( '-', '_', $this->options['form_name'] );

	}


	/**
	 * Обработка входной информации.
	 *
	 * @param void
	 * @return void
	 */ 
	protected function input_data_processing() {

		// Запись типов тега input
		$this->input['type'] = array('button', 'checkbox', 'file', 'image', 'radio', 'reset', 'submit', 'text' );
		// Запись расширенных типов тега input
		$this->input['add_type'] = array('password', 'date', 'datetime', 'email', 'number', 'range', 'search', 'tel', 'time', 'url', 'month', 'week', 'hidden');

		// Получение префикса имен полей
		$form_name_prefix = $this->options['form_name'] . '-';
		// Обработка POST данных и запись в локальный массив
		$this->post_data_handling($form_name_prefix);
	}


	/**
	 * Обработка POST данных и запись в локальный массив.
	 *
	 * @param void
	 * @return void
	 */
	private function post_data_handling($form_name_prefix) {

		// Проверка массива POST
		if( !isset($_POST) || empty($_POST) ) {
			throw new Exception('Empty post data!' );
		}

		// Резервирование данных
		foreach( $_POST as $key => $value ) {
			// Удаление префикса из имени поля
			$short_key = explode($form_name_prefix, $key);
			// Возможны ли тут исключения? Все что без префикса - лесом.
			if( isset($short_key[1]) ) {
				$short_key = $short_key[1];
				// Отсеивание технических данных
				if ($short_key != 'question' && $short_key != 'form-name' ) {
					// Запись значений полей в локальный массив
					$this->post_data[$short_key] = htmlspecialchars($value, ENT_NOQUOTES, 'UTF-8');
				}
			}
		}
		if( !isset($this->static_form) || !$this->static_form ) {
			unset($_POST);
		}
	}


	/**
	 * Подключение файла с текстовой информацией.
	 * 
	 * Массив содержит названия полей и подсказки к полям.
	 * Информация сортирована тематически по блокам.
	 *
	 * @param void
	 * @return void
	 */
	private function get_txt_data() {

		// Проверка наличия файла с названиями полей
		$file = '/conf/fields.php';

		// Пути
		$path = ['theme'=>TEMPLATEPATH];
		// Пути к плагинам
		$path = array_merge_recursive($path,$this->plugins_path);

		// Итерация путей
		foreach( $path as $key => $sub_path ) {
			$file_path = $sub_path . $file;
			// Исключение (нет основного файла)
			if( $key == 'theme' && !file_exists($file_path) ) {
				throw new Exception('File error: ' . $file_path);
			}

			// Подклчение файла с именами полей.
			if( file_exists($file_path) ) {
				require_once $file_path;
			}
		}

		/**
		 * Сортировка згруженного массива по назначению.
		 * Результатом работы будет два массива (Названий полей и подсказок) с одинаковыми индексами.
		 */
		foreach( $fields as $names_block => $text_data ) {
			// Выделение имани и суффикса блока
			$tmp = explode('_', $names_block);
			$block_name = $tmp[0];
			$block_suffix = $tmp[1];

			foreach( $text_data as $field_name => $value ) {
				// Имя свойства равно имени суффикса labels/placeholders
				$this->{$block_suffix}[$block_name][$field_name] = $value;
			}
		}
	}


	/**
	 * Параметры формы.
	 * 
	 * Подключение конфигурационного файла с параметрами форм.
	 * Выбор массива происходит по переданному имени формы.
	 *
	 * @param void
	 * @return void
	 */
	private function get_form_data() {

		// Проверка наличия файла с параметрами форм
		$file = '/conf/forms.php';

		// Пути
		$path = ['theme'=>TEMPLATEPATH];
		// Пути к плагинам
		$path = array_merge_recursive($path,$this->plugins_path);

		// Итерация путей
		foreach( $path as $key => $sub_path ) {
			$file_path = $sub_path . $file;
			// Исключение (нет основного файла)
			if( $key == 'theme' && !file_exists($file_path) ) {
				throw new Exception('File error: ' . $file_path);
			}

			// Подклчение файла с именами полей.
			if( file_exists($file_path) ) {
				require_once $file_path;
			}

		}

		// Проверка наличия параметров контактной формы
		if( !isset(${$this->options['array_name']}) || empty(${$this->options['array_name']}) ) {
			throw new Exception('There is no form parametrs! Array: ' . $this->options['array_name']);
		}

		// Запись заголовкаов контактной формы
		foreach(${$this->options['array_name']}['titles'] as $key => $value) {
			$this->options[$key] = $value;
		}

		// Удаление элементов не являющимися описанием полей формы (заголовки)
		unset(${$this->options['array_name']}['titles']);

		// Запись параметров формы как свойства объекта
		$this->fields = ${$this->options['array_name']};

		// Проход циклом по массиву параметров для восстановления параметров по умолчанию
		foreach( $this->fields as $name => $options ) {
			// Нет массива параметров поля
			if( !is_array($options) ) {
				// Запись значения по умолчанию
				$this->fields[$name] = [];
			}
			// Нет параметра типа поля
			if( !isset($options['type']) ) {
				// Запись значения по умолчанию
				$this->fields[$name]['type'] = $defaults['type'];
			}
		}
	}


	/**
	 * Преобразование данных в необходимый вид.
	 * 
	 * Сборка данных в единое целое.
	 * В зависимости от ситуации callback функции могут собирать html или массивы данных.
	 *
	 * @param array $incoming_data
	 * @param calback $header_callback
	 * @param calback $line_callback
	 * @param calback $wrapper_callback
	 * @return mixed
	 */
	private function get_result_set( $incoming_data, $header_callback, $line_callback, $wrapper_callback ) {

		// Массив для сортировки полей по блокам
		// Каждый отдельный блок записывется в элемент массива
		$result_set = [];

		// (1) Проход цикла по массиву входных даннх
		foreach( $incoming_data as $incoming_name => $incoming_value ) {
			// (2) Проход цикла по массивам текстовых значений (названий полей) для поиска совпадений индексов массивов
			foreach( $this->labels as $block_name => $text_data ) {
				// (3) Проход цикла по массиву названий полей для поиска совпадений
				foreach( $text_data as $lable_name => $label ) {
					// Совпадение индексов массивов пост данных и имен полей
					if( $incoming_name == $lable_name ) {
						// Проверка сущесвования временной переменной (первый проход цикла или нет)
						if( !isset( $result_set[$block_name] ) ) {
							// Создание элемента массива для хранения блока полей
							$result_set[$block_name] = '';
							if( $this->fields[$lable_name]['type'] != 'hidden' ) {
								// Запись заголовка
								$result_set[$block_name] .= call_user_func( $header_callback, $text_data['block_title'] );
							}
						}
						// Запись значения
						$result_set[$block_name] .= call_user_func( $line_callback, $incoming_value, $label, $lable_name, $block_name );
					} // end if
				} // end foreach (3)
			} // end foreach (2)
		} // end foreach (1)

		// Оформление результата
		return call_user_func( $wrapper_callback, $result_set );
	}


} // end FormsData
?>