/**
 * Google map events
 * 
 *
 * @author veselka.ua
 * @version 0.2
 * @modify 22-03-2018
 */

var gMapEvents = (function ($)
{
	$.mapEvents =
	{

		/**
		 * Map data
		 */
		gMaps:{
			contacts: {
				id: 'map-contacts',
				options: {
					single: true,
					popup: true,
					ajax: false,
				},
			},
			labeledcontacts: {
				id: 'map-contacts-offset',
				options: {
					single: true,
					popup: true,
					ajax: false,
				},
			},
			multicontacts: {
				id: 'map-contacts-multiple',
				options: {
					single: false,
					popup: true,
					ajax: false,
				},
			},
			realty: {
				id: 'map-full-realty',
				options: {
					single: false,
					delayed: false,
					ajax: true,
				},
			},
			search: {
				id: 'map-full-search',
				options: {
					single: false,
					delayed: true,
					ajax: true,
				},
			},
			district: {
				id: 'map-full-district',
				options: {
					single: false,
					delayed: true,
					ajax: true,
				},
			},
			single: {
				id: 'single-map',
				options: {
					single: true,
					popup: false,
					ajax: true,
				},
			},
		},


		/**
		 * Verification codes
		 */
		proof: {
			response:	{
				realty:		'-e@Jv2Tp[,7:kHtE&s8C1y$W/jE',
				search:		'q7J$m3F@c9L#v3F*,4L^f4K!u1N)',
				district:	'pM(1dU$54t@B6zA-0dP3%kE3d/w',
			},
			answer:		{
				realty:		'9kT?sP6OE^mL*f4Wq1N{Su#6tiG',
				search:		'3jk$R3dc(j#gs$q%$9sFz2mA3?o',
				district:	'+jW6k!@l9E%v7Uh&2Eh1@4KJ6^5',
			},
			contacts:	'9dfa89137136178c5721abd9b7c2ea62',
		},


		/**
		 * Default options
		 */
		defaults: {
			ajax_file:				'/ajax/search_form.php',
			icon:					template_dir+'/assets/img/map_marker.svg',
			zoom:					15,
			listContainer:			'nb-list-content',
			searchFormContainer:	'affix-container',
			serchFormPinch:			'landing-map-pinch',
			bodyClass:				'search-on-map',
		},


		/**
		 * Permits multiple Ajax requests
		 */
		deferred: [],


		/**
		 * Map Options
		 */
		mapOptions: {
			draggable: true,
			scrollwheel: false,
			scaleControl: true,
			mapTypeId: google.maps.MapTypeId.ROADMAP,
			mapTypeControl: false,
			zoomControlOptions: {
				position: google.maps.ControlPosition.LEFT_BOTTOM
			},
			streetViewControlOptions: {
				position: google.maps.ControlPosition.LEFT_BOTTOM
			},
		},


		/**
		 * Display dimensions
		 */
		dim: {
			height: '',
			width: '',
		},


		/**
		 * Init
		 */
		init: function ()
		{

			// Searchform common actions
			this.commonActions();

			// Searchform common actions
			this.homePageMap();

			// Initialize maps
			this.gMapsConstruct();

			// Behavior actions
			this.bahaviorActions();
		},


		/**
		 * Common listeners
		 */
		commonActions: function()
		{
			var _this = this;

			// Dimensions
			_this.dim.width = $( window ).width();
			_this.dim.height = $( window ).height();

			// Realty plugin is on?
			if( typeof realty_dir !== 'undefined' )
			{
				this.defaults.ajax_file = realty_dir + this.defaults.ajax_file;
			}

			// Resize
			$( window ).resize( function() {
				// Only width
				if( _this.dim.width != $( window ).width() )
				{
					// New width
					_this.dim.width = $( window ).width();
					// Maps reload
					$.each( _this.gMaps, function( index, element )
					{
						_this.reloadMap(index);
					});
				}
				// Only height
				if( _this.dim.height != $( window ).height() )
				{
					// New height
					_this.dim.height = $( window ).height();
					// Maps hight reload
					_this.setMapHight();
				}
			});
		},


		/**
		 * Fix search form on home page map
		 */
		homePageMap: function ()
		{
			if( this.gMaps.realty == undefined )
			{
				return true;
			}
			// Set margin
			this.setMapHight();
		},


		/**
		 * Initialize maps
		 */
		gMapsConstruct: function ()
		{
			// Content construct methods
			this.setContent();

			var _this = this;

			// Unset nonexistent elements
			$.each( this.gMaps, function( index, element )
			{
				// No map container
				if( $('#'+element.id).size() == 0 )
				{
					delete _this.gMaps[index];
					return;
				}
			});

			// Construct maps
			$.each( this.gMaps, function( index, element )
			{
				// Maps: delayed load marker
				if( _this.gMaps[index].options.delayed )
				{
					return;
				}
				_this.loadMap(index);
			});
		},


		/**
		 * Behavior listeners
		 */
		bahaviorActions: function()
		{
			var _this = this;

			// Change view
			$( '#realty-on-map' ).on( 'click', function()
			{
				$.runRealtySearchEvents.searchFormLoad();
				// Switch map/list
				_this.switchView(false);
				// Set map hight
				_this.setMapHight();
				// Load map
				_this.delayedLoadMap();
				
			});
			$( '#realty-list' ).on( 'click', function()
			{
				// Switch map/list
				_this.switchView(true);
				// Set map hight
				_this.setMapHight();
			});
		},


////////////////////////////// COMMON MTHODS //////////////////////////////


		/**
		 * Map delayed load
		 */
		delayedLoadMap: function()
		{
			var _this = this;

			// Construct maps
			$.each( this.gMaps, function( index, element )
			{
				// Maps: delayed load marker
				if( !_this.gMaps[index].options.delayed )
				{
					return;
				}
				_this.loadMap(index);
			});
		},


		/**
		 * Map load
		 * 
		 * @argument bool index
		 */
		loadMap: function(index)
		{
			var _this = this;

			// Already exist obj check
			if( $.type(this.gMaps[index].map) == 'object')
			{
				return true;
			}

			// Get ajax data
			this.getMarkerData(index);

			if( this.gMaps[index].options.ajax )
			{
				// Ajax loaded?
				$.when.apply($, this.deferred).done(function ()
				{
					// Load map obj
					_this.loadMapObj(index);
					// Register map
					_this.registerMaps(index);
				});
			}
			else
			{
				// Load map obj
				this.loadMapObj(index);
				// Register map
				this.registerMaps(index);
			}
		},


		/**
		 * Map reload
		 * 
		 * @argument bool index
		 */
		reloadMap: function(index)
		{

			// Nonexistent obj check
			if( $.type(this.gMaps[index].map) != 'object')
			{
				return true;
			}

			// the 'map' here is the result of the created map obj
			google.maps.event.trigger( this.gMaps[index].map, 'resize' );
			// Single maps: reload center 
			if( this.gMaps[index].options.single )
			{
				if( typeof this.gMaps[index].marker.offset != 'undefined' )
				{
					var offset = this.gMaps[index].marker.offset;
					offset = parseFloat(offset);
					if( this.dim.width > 767 ) {
						this.gMaps[index].map.setCenter( new google.maps.LatLng(this.gMaps[index].marker.lat, this.gMaps[index].marker.lng + offset) );
					}
					else
					{
						this.gMaps[index].map.setCenter( new google.maps.LatLng(this.gMaps[index].marker.lat + offset/2, this.gMaps[index].marker.lng) );
					}
				}
				else
				{
					this.gMaps[index].map.setCenter( this.gMaps[index].latLng );
				}
			// Multiple map: fit bounds
			}
			else
			{
				this.gMaps[index].map.fitBounds( this.gMaps[index].bounds );
			}
		},


		/**
		 * Map div height
		 */
		setMapHight: function()
		{
			var _this = this;
			var ids = ['search', 'district', 'realty'];
			// Search wrapper height
			var searchHeight = $('.'+this.defaults.listContainer).outerHeight();
			// Navbar-top height fix for map slide
			var wrapHeight = 0;
			if( $('#navbar-top').length > 0 )
			{
				wrapHeight = $('#navbar-top').outerHeight();
			}
			// Navbar-top margin-top fix for map slide
			var wrapMargin = 0;
			if( $('#navbar-landing-fixed').length > 0 )
			{
				wrapMargin = $('#navbar-landing-fixed').outerHeight();
				searchHeight = searchHeight + wrapMargin;
			}
			//console.log(this.dim.height);
			//console.log(wrapMargin);
			// Window height less that searchform listContainer
			// Only for homepage
			if(  $('#map-full-realty').length > 0 && searchHeight > this.dim.height )
			{
				this.dim.height = searchHeight;
			}

			// Set height
			$.each( ids, function( i, index )
			{
				// Is obj set?
				if( _this.gMaps[index] == undefined )
				{
					return;
				}
				// Set height
				if( $('#'+_this.gMaps[index].id).length > 0 && $('#'+_this.gMaps[index].id).is(':visible') )
				{
					$('#'+_this.gMaps[index].id).css('height', (_this.dim.height-wrapMargin)+'px');
					$('.'+_this.defaults.listContainer).css('margin-top', '-'+(_this.dim.height-wrapMargin)+'px');
					$('#wrap').css('min-height', (_this.dim.height)+wrapHeight+'px');
					_this.reloadMap(index);
				}
				else
				{
					$('.'+_this.defaults.listContainer).css('margin-top', 'inherit');
					$('#wrap').css('min-height', '100%');
				}
			});
			//console.log(this.dim.height);
		},


		/**
		 * Switch list/map
		 * 
		 * @argument bool list
		 */
		switchView: function(list)
		{
			if( this.gMaps.search == undefined && this.gMaps.district == undefined )
			{
				return true;
			}
			if( list )
			{
				$('#realty-on-map').removeClass('active');
				$('#realty-list').addClass('active');
				$.runRealtySearchEvents.affixLoad();
				$('body').removeClass('search-on-map');
			}
			else
			{
				$('#realty-on-map').addClass('active');
				$('#realty-list').removeClass('active');
				$.runRealtySearchEvents.affixDestroy();
				$('body').addClass('search-on-map');
				// Scroll to top
				var _this = this;
				var ids = ['search','district'];
				$.each( ids, function( i, index )
				{
					// Is obj set?
					if( _this.gMaps[index] == undefined ) {
						return;
					}
					$('html, body').animate({
					  scrollTop: $('#'+_this.gMaps[index].id).offset().top
					}, 1000);
				});
			}
		},



////////////////////////////// ALLOCATION MAP OPTIONS //////////////////////////////


		/**
		 * Popups content construct
		 */
		setContent: function()
		{
			var _this = this;
			/**
			 * Contacts popup content
			 */
			this.gMaps.contacts.popup = function()
			{
				return _this.contactsContent('contacts');
			};
			/**
			 * Contacts popup content
			 */
			this.gMaps.labeledcontacts.popup = function()
			{
				return _this.contactsContent('labeledcontacts');
			};
			/**
			 * Home page map popup content
			 * 
			 * @argument string index
			 * @argument int i
			 */
			this.gMaps.multicontacts.popup = function(index, i)
			{
				return _this.contactsMultipleContent(index, i);
			};
			/**
			 * Home page map popup content
			 * 
			 * @argument string index
			 * @argument int i
			 */
			this.gMaps.realty.popup = function(index, i)
			{
				return _this.realtyContent(index, i);
			};
			/**
			 * Search page map popup content
			 * 
			 * @argument string index
			 * @argument int i
			 */
			this.gMaps.search.popup = function(index, i)
			{
				return _this.realtyContent(index, i);
			};
			/**
			 * Districts page map popup content
			 * 
			 * @argument string index
			 * @argument int i
			 */
			this.gMaps.district.popup = function(index, i)
			{
				return _this.realtyContent(index, i);
			};
		},


////////////////////////////// REGISTER MAPS METHODS //////////////////////////////


		/**
		 * Get map data.
		 * 
		 * @argument string index
		 */
		getMarkerData: function(index)
		{
			var _this = this;

			if( this.gMaps[index].options.ajax )
			{
				var data = {};
				data.proof = this.proof.response[index];
				data.district = $('section').data('district');
				if( data.district == undefined )
				{
					data.district = 0;
				}

				// Get data
				this.deferred.push(
					$.ajax({
						url: this.defaults.ajax_file,
						type: 'POST',
						dataType: 'json',
						data: data,
						//async: false,
						success: function(response, code)
						{
							//console.log(response);
							if( response.proof != this.proof.answer[index] )
							{
								return;
							}
							this.gMaps[index].markers = response.data;
							//console.log(response.data);
							//console.log(this.gMaps[index].markers);
						}.bind(this),
						error: function(response, code)
						{
							//console.log(code+' '+response);
						},
					})
				);
			}
			else
			{
				// Single maps: set center via options
				if( this.gMaps[index].options.single )
				{
					// Get data attributes
					this.gMaps[index].marker = $('#'+this.gMaps[index].id).data();
					//this.gMaps[index].marker.title = $('h1').html();
					return true;
				}
				else
				{
					this.gMaps[index].markers = [];
					$('#'+this.gMaps[index].id+' div').each(function( i, item ) {
						// Get data attributes
						_this.gMaps[index].markers[i] = $(item).data();
					});

				}
			}
		},


		/**
		 * Load google map.
		 * 
		 * @argument string index
		 */
		loadMapObj: function(index)
		{
			// Map options
			var mapOptions = this.mapOptions;
			// Single maps: set center via options
			if( this.gMaps[index].options.single )
			{
				// Coordinats
				this.gMaps[index].latLng = new google.maps.LatLng(this.gMaps[index].marker.lat, this.gMaps[index].marker.lng);
				mapOptions.zoom = this.defaults.zoom;
				if( typeof this.gMaps[index].marker.zoom != 'undefined' )
				{
					mapOptions.zoom = this.gMaps[index].marker.zoom;
				}
				if( typeof this.gMaps[index].marker.offset != 'undefined' )
				{
					var offset = this.gMaps[index].marker.offset;
					offset = parseFloat(offset);
					if( this.dim.width > 767 ) {
						mapOptions.center = new google.maps.LatLng(this.gMaps[index].marker.lat, this.gMaps[index].marker.lng + offset);
					}
					else
					{
						mapOptions.center = new google.maps.LatLng(this.gMaps[index].marker.lat + offset/2, this.gMaps[index].marker.lng);
					}
				}
				else
				{
					mapOptions.center = this.gMaps[index].latLng;
				}
				//mapOptions.styles = this.styles;
			}

			// Map
			this.gMaps[index].map = new google.maps.Map(document.getElementById(this.gMaps[index].id), mapOptions);
		},


		/**
		 * Load google map.
		 * 
		 * @argument string index
		 */
		registerMaps: function(index)
		{
			var _this = this;

			// Info window
			var infowindow = new google.maps.InfoWindow();

			// Single maps: load marker
			if( this.gMaps[index].options.single )
			{
				// Default icon
				if( this.gMaps[index].marker.icon == undefined )
				{
					this.gMaps[index].marker.icon = this.defaults.icon;
				}
				// Marker
				var marker = new google.maps.Marker({
					position: this.gMaps[index].latLng,
					map: this.gMaps[index].map,
					icon: this.gMaps[index].marker.icon,
					title: this.gMaps[index].marker.title,
					address: this.gMaps[index].marker.address,
				});
				// Popup
				if( this.gMaps[index].options.popup )
				{
					google.maps.event.addListener(marker, 'click', function()
					{
						infowindow.close();
						infowindow.setContent( _this.gMaps[index].popup() );
						infowindow.open( _this.gMaps[index].map, marker );
					});
				}

			} else {

				this.gMaps[index].bounds = new google.maps.LatLngBounds();

				for( i = 0; i < this.gMaps[index].markers.length; i++ )
				{

					var marker = new google.maps.Marker({
						position: new google.maps.LatLng(this.gMaps[index].markers[i].lat, this.gMaps[index].markers[i].lng),
						map: this.gMaps[index].map,
						icon: this.defaults.icon,
						title: this.gMaps[index].markers[i].title,
						address: this.gMaps[index].markers[i].address,
					});
					//console.log(marker);
					this.gMaps[index].bounds.extend(marker.position);

					google.maps.event.addListener( marker, 'click', (function (marker, i)
					{
						return function ()
						{
							infowindow.close();
							infowindow.setContent( _this.gMaps[index].popup(index, i) );
							infowindow.open( _this.gMaps[index].map, marker );
						}
					})(marker, i));
				};
				//console.log(this.gMaps[index].bounds);
				this.gMaps[index].map.fitBounds(this.gMaps[index].bounds);

				/*var listener = google.maps.event.addListener(map, "idle", function () {
					map.setZoom(11);
					google.maps.event.removeListener(listener);
				});*/
			}
			google.maps.event.addListener( this.gMaps[index].map, "click", function(event)
			{
				infowindow.close();
			});
		},


		/**
		 * Contacts popup content
		 */
		contactsContent: function(index)
		{
			var contentString =
			'<div class="scrollFix">'+
				'<div id="siteNotice"></div>';
				if( this.gMaps[index].marker.title )
				{
					contentString = contentString+
					'<h1>'+this.gMaps[index].marker.title+'</h1>';
				}
				contentString = contentString+
				'<div id="bodyContent">';
					if( this.gMaps[index].marker.description )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-info fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].marker.description+
						'</p>';
					}
					if( this.gMaps[index].marker.phone )
					{
						var phone = this.gMaps[index].marker.phone;
							phone = phone.replace(/\ /g, '');
							phone = phone.replace(/\(/g, '');
							phone = phone.replace(/\)/g, '');
							phone = phone.replace(/\-/g, '');
							phone = phone.replace(/\_/g, '');
							phone = phone.replace('+38', '');
						var phoneProper = '('+
									phone.substring(0, 3)+
									') '+
									phone.substring(3, 6)+
									'-'+
									phone.substring(6, 8)+
									'-'+
									phone.substring(8, 10);
							phone = '+38' + phone;
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a href="tel:'+phone+'">'+
								'<i class="fa fa-phone fa-fw" aria-hidden="true"></i>&nbsp;'+phoneProper+
							'</a>'+
						'</p>';
					}
					if( this.gMaps[index].marker.email )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a href="mailto:'+this.gMaps[index].marker.email+'">'+
								'<i class="fa fa-envelope fa-fw" aria-hidden="true"></i>&nbsp;'+
								this.gMaps[index].marker.email+
							'</a>'+
						'</p>';
					}
					if( this.gMaps[index].marker.name )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-flag fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].marker.name+
						'</p>';
					}
					if( this.gMaps[index].marker.address )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-map-marker fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].marker.address+
						'</p>';
					}
					if( this.gMaps[index].marker.hours )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-clock-o fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].marker.hours+
						'</p>';
					}
				contentString = contentString+
				'</div>'+
			'</div>';
			return contentString;
		},




		/**
		 * Multiple contacts
		 * 
		 * @argument string index
		 * @argument int i
		 */
		contactsMultipleContent: function(index, i)
		{
			var contentString =
			'<div class="scrollFix">'+
				'<div id="siteNotice"></div>';
				if( this.gMaps[index].markers[i].title )
				{
					contentString = contentString+
					'<h1>'+this.gMaps[index].markers[i].title+'</h1>';
				}
				contentString = contentString+
				'<div id="bodyContent">';

					if( this.gMaps[index].markers[i].description )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-info fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].description+
						'</p>';
					}
					if( this.gMaps[index].markers[i].phone )
					{
						var phone = this.gMaps[index].markers[i].phone;
							phone = phone.replace(/\ /g, '');
							phone = phone.replace(/\(/g, '');
							phone = phone.replace(/\)/g, '');
							phone = phone.replace(/\-/g, '');
							phone = phone.replace(/\_/g, '');
							phone = phone.replace('+38', '');
						var phoneProper = '('+
									phone.substring(0, 3)+
									') '+
									phone.substring(3, 6)+
									'-'+
									phone.substring(6, 8)+
									'-'+
									phone.substring(8, 10);
							phone = '+38' + phone;
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a href="tel:'+phone+'">'+
								'<i class="fa fa-phone fa-fw" aria-hidden="true"></i>&nbsp;'+phoneProper+
							'</a>'+
						'</p>';
					}
					if( this.gMaps[index].markers[i].email )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a href="mailto:'+this.gMaps[index].markers[i].email+'">'+
								'<i class="fa fa-envelope fa-fw" aria-hidden="true"></i>&nbsp;'+
								this.gMaps[index].marker.email+
							'</a>'+
						'</p>';
					}
					if( this.gMaps[index].markers[i].name )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a href="'+this.gMaps[index].markers[i].name+'">'+
								'<i class="fa fa-flag fa-fw" aria-hidden="true"></i>&nbsp;'+
								this.gMaps[index].markers[i].name+
							'</a>'+
						'</p>';
					}
					if( this.gMaps[index].markers[i].address )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-map-marker fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].address+
						'</p>';
					}
					if( this.gMaps[index].markers[i].hours )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-clock-o fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].hours+
						'</p>';
					}
					if( this.gMaps[index].markers[i].url )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a rel="noindex" href="'+this.gMaps[index].markers[i].url+'">'+
								'<i class="fa fa-link fa-fw" aria-hidden="true"></i>&nbsp;Узнать подробнее'+
							'</a>'+
						'</p>';
					}
				contentString = contentString+
				'</div>'+
			'</div>';
			return contentString;
		},



		/**
		 * Realty list popups content construct
		 * 
		 * @argument string index
		 * @argument int i
		 */
		realtyContent: function(index, i)
		{
			var contentString =
			'<div class="scrollFix">'+
				'<div id="siteNotice"></div>';
				if( this.gMaps[index].markers[i].title )
				{
					contentString = contentString+
					'<h1>'+this.gMaps[index].markers[i].title+'</h1>';
				}
				contentString = contentString+
				'<div id="bodyContent">';
					if( this.gMaps[index].markers[i].address )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-map-marker fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].address+
						'</p>';
					}
					if( this.gMaps[index].markers[i].builder )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-flag fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].builder+
						'</p>';
					}
					if( this.gMaps[index].markers[i].stage )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-line-chart fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].stage+
						'</p>';
					}
					if( this.gMaps[index].markers[i].deadline )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-calendar fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].deadline+
						'</p>';
					}
					if( this.gMaps[index].markers[i].price_m2 )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-shopping-cart fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].price_m2+
						'</p>';
					}
					if( this.gMaps[index].markers[i].price )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<i class="fa fa-shopping-cart fa-fw" aria-hidden="true"></i>&nbsp;'+
							this.gMaps[index].markers[i].price+
						'</p>';
					}
					if( this.gMaps[index].markers[i].url )
					{
						contentString = contentString+
						'<p class="google-marker-info">'+
							'<a href="'+this.gMaps[index].markers[i].url+'">'+
								'<i class="fa fa-link fa-fw" aria-hidden="true"></i>&nbsp;Узнать подробнее'+
							'</a>'+
						'</p>';
					}
				contentString = contentString+
				'</div>'+
			'</div>';
			return contentString;
		},


	};

})(jQuery);

window.mapInitialize = function()
{
	$.mapEvents.init();
}
google.maps.event.addDomListener(window, 'load', mapInitialize);